//========================================================================
//
// config.h
//
// Copyright 1996-2014 Glyph & Cog, LLC
// Copyright 2018-2026 Adam Sampson <ats@offog.org>
//
//========================================================================

#ifndef CONFIG_H
#define CONFIG_H

#include <memory>
#ifdef NEED_OPTIONAL
#include <optional>
#endif
#include <string>

#include <goo/GooString.h>

//------------------------------------------------------------------------
// version
//------------------------------------------------------------------------

// version
#define xpoppleVersion          "3.04"

// copyright notice
#define xpoppleCopyright "Copyright 1996-2014 Glyph & Cog, LLC\n" \
    "Copyright 2014-2026 xpopple contributors"

//------------------------------------------------------------------------
// paper size
//------------------------------------------------------------------------

// default paper size (in points) for PostScript output
#define defPaperWidth  612    // American letter (8.5x11")
#define defPaperHeight 792

//------------------------------------------------------------------------
// config file (xpdfrc) path
//------------------------------------------------------------------------

// user config file name, relative to the user's home directory
#define xpoppleUserConfigFile ".xpdfrc"

// system config file name (set via the configure script)
#define xpoppleSysConfigFile SYSTEM_XPDFRC

//------------------------------------------------------------------------
// X-related constants
//------------------------------------------------------------------------

// default maximum size of color cube to allocate
#define defaultRGBCube 5

//------------------------------------------------------------------------
// Poppler backwards compatibility
//------------------------------------------------------------------------

// Poppler has fixed various const correctness problems in the xpdf API;
// 0.64.0 in particular introduced a number of new constraints. In many
// cases xpopple's code can be written to work either way, but where
// this isn't possible, PCONST means "const after 0.64.0".
#ifdef POPPLER_64_CONST_API
#define PCONST const
#else
#define PCONST
#endif

// From 0.86.0, some methods started returning std::string or GooString rather
// than (non-owning) GooString *. toString adapts these to std::string.
static inline std::string toString(const std::string &s) {
  return s;
}
#ifdef GOOSTRING_IS_STRING
static inline std::string toString(const GooString &s) {
  return s.toStr();
}
static inline std::string toString(const GooString *s) {
  return s->toStr();
}
#else
static inline std::string toString(PCONST GooString &s) {
  return std::string(s.getCString(), s.getLength());
}
static inline std::string toString(PCONST GooString *s) {
  return std::string(s->getCString(), s->getLength());
}
#endif

// Make an owning GooString pointer or unique_ptr from a std::string.
#ifdef GOOSTRING_IS_STRING
static inline GooString *makeGooString(const std::string &s) {
  return new GooString(s);
}
#else
static inline GooString *makeGooString(const std::string &s) {
  return new GooString(s.c_str(), s.size());
}
#endif
static inline GooString *makeGooString(const std::string *s) {
  if (!s) {
    return NULL;
  }
  return makeGooString(*s);
}
static inline std::unique_ptr<GooString>
makeGooStringPtr(const std::string &s) {
  return std::unique_ptr<GooString>(makeGooString(s));
}
static inline std::unique_ptr<GooString>
makeGooStringPtr(const std::string *s) {
  return std::unique_ptr<GooString>(makeGooString(s));
}

#ifdef NEED_OPTIONAL
// Make an optional<GooString> from a possibly-NULL std::string *.
static inline std::optional<GooString>
makeGooStringOpt(const std::string *s) {
  if (!s) {
    return std::nullopt;
  }
  return std::make_optional<GooString>(s->c_str(), s->size());
}
#endif

#endif
